
import React from 'react';
import { Head, useForm, usePage } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';

interface ClientLicense {
    id: number;
    license_key: string;
    status: string;
    domain?: string | null;
    message?: string | null;
    activated_at?: string | null;
    last_verified_at?: string | null;
}

interface SetupProps {
    current?: ClientLicense | null;
    licenseError?: string | null;
    [key: string]: unknown;
}

const Index: React.FC = () => {
    const { current, licenseError } = usePage<SetupProps>().props;

    return (
        <>
            <Head title="Pengaturan Lisensi" />
            <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-slate-900 via-slate-950 to-slate-900 text-foreground px-4">
                <div className="w-full max-w-md">
                    <div className="text-center mb-6">
                        <h1 className="text-xl font-semibold text-slate-50">
                            Aktivasi Lisensi
                        </h1>
                        <p className="text-xs text-slate-300 mt-1">
                            Masukkan license key yang Anda dapatkan saat pembelian untuk mengaktifkan aplikasi.
                        </p>
                    </div>
                    <div className="border border-slate-700/80 bg-slate-900/70 backdrop-blur-sm text-card-foreground shadow-xl rounded-2xl p-5 space-y-4">
                        {current && (
                            <div className="rounded-lg border border-slate-700 bg-slate-900/80 px-3 py-3 text-xs space-y-1">
                                <div className="flex items-center justify-between gap-2">
                                    <span className="text-slate-300">
                                        Status saat ini:
                                    </span>
                                    <StatusBadge status={current.status} />
                                </div>

                                {current.message && (
                                    <div className="text-slate-300">
                                        {current.message}
                                    </div>
                                )}
                                <div className="flex flex-col gap-1 text-[11px] text-slate-400">
                                    {current.domain && (
                                        <span>
                                            Domain terdaftar:{' '}
                                            <span className="font-mono text-slate-200">
                                                {current.domain}
                                            </span>
                                        </span>
                                    )}
                                    {current.activated_at && (
                                        <span>
                                            Diaktivasi:{' '}
                                            {new Date(current.activated_at).toLocaleString('id-ID')}
                                        </span>
                                    )}
                                    {current.last_verified_at && (
                                        <span>
                                            Terakhir dicek:{' '}
                                            {new Date(current.last_verified_at).toLocaleString('id-ID')}
                                        </span>
                                    )}
                                </div>
                            </div>
                        )}
                        {licenseError && (
                            <div className="rounded-lg border border-red-500/60 bg-red-500/10 px-3 py-2 text-xs text-red-100">
                                {licenseError}
                            </div>
                        )}
                        <form onSubmit={handleSubmit} className="space-y-4">
                            <div className="space-y-1">
                                <label className="text-sm font-medium text-slate-100">
                                    License Key
                                </label>
                                <Input
                                    value={data.license_key}
                                    onChange={(e) => setData('license_key', e.target.value)}
                                    placeholder="Contoh: WIOOS-XXXX-XXXX-XXXX"
                                    className="font-mono text-xs bg-slate-900/80 border-slate-700 text-slate-100 placeholder:text-slate-500"
                                />
                                {errors.license_key && (
                                    <p className="text-xs text-red-300 mt-1">
                                        {errors.license_key}
                                    </p>
                                )}
                            </div>
                            <Button
                                type="submit"
                                disabled={processing}
                                className="w-full"
                            >
                                {current ? 'Update Lisensi' : 'Aktifkan Lisensi'}
                            </Button>

                            <p className="text-[10px] text-slate-500 text-center">
                                Lisensi ini hanya berlaku untuk satu instalasi. Jika terjadi masalah aktivasi,
                                silakan hubungi support Anda.
                            </p>
                        </form>
                    </div>
                </div>
            </div>
        </>
    );
};

interface StatusBadgeProps {
    status?: string | null;
}

const StatusBadge: React.FC<StatusBadgeProps> = ({ status }) => {
    const value = (status || 'inactive').toLowerCase();

    let variantClass = '';
    if (value === 'active') variantClass = 'bg-emerald-500/15 text-emerald-300 border border-emerald-500/40';
    else if (value === 'disabled') variantClass = 'bg-slate-500/15 text-slate-200 border border-slate-500/40';
    else if (value === 'expired') variantClass = 'bg-red-500/15 text-red-200 border border-red-500/40';
    else variantClass = 'bg-amber-500/15 text-amber-200 border border-amber-500/40';

    return (
        <Badge
            className={`text-[11px] font-semibold px-2 py-0.5 ${variantClass}`}
        >
            {value.toUpperCase()}
        </Badge>
    );
};

(Index as any).layout = (page: React.ReactNode) => page;

export default Index;
