<?php

namespace App\Services;

use App\Models\ClientLicense;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class LicenseClientService
{
    public function getCurrent(): ?ClientLicense
    {
        return ClientLicense::query()->latest()->first();
    }

    public function activate(string $licenseKey): array
    {
        $baseUrl    = config('license_client.api_base_url');
        $domain     = request()->getHost();
        $env        = config('license_client.environment', 'production');

        try {
            $response = Http::timeout(10)
                ->acceptJson()
                ->post($baseUrl . '/license/activate', [
                    'license_key' => $licenseKey,
                    'domain'      => $domain,
                    'environment' => $env,
                ]);

            $data = $response->json();

            if (! $response->successful() || ($data['status'] ?? null) !== 'success') {
                return [
                    'ok'      => false,
                    'status'  => $data['code'] ?? 'ERROR',
                    'message' => $data['message'] ?? 'Gagal mengaktivasi lisensi.',
                ];
            }

            $license = ClientLicense::updateOrCreate(
                ['license_key' => $licenseKey],
                [
                    'status'          => $data['data']['status'] ?? 'active',
                    'domain'          => $data['data']['domain'] ?? $domain,
                    'message'         => $data['message'] ?? 'Lisensi berhasil diaktivasi.',
                    'activated_at'    => now(),
                    'last_verified_at'=> now(),
                ]
            );

            return [
                'ok'      => true,
                'status'  => $license->status,
                'message' => $license->message,
            ];
        } catch (\Throwable $e) {
            Log::error('License activation error', ['error' => $e->getMessage()]);

            return [
                'ok'      => false,
                'status'  => 'NETWORK_ERROR',
                'message' => 'Gagal menghubungi server lisensi. Coba beberapa saat lagi.',
            ];
        }
    }

    public function verify(?ClientLicense $license = null, bool $force = false): array
    {
        $license ??= $this->getCurrent();

        if (!$license) {
            return [
                'ok'      => false,
                'status'  => 'NO_LICENSE',
                'message' => 'Lisensi belum diatur.',
            ];
        }

        if (! $force && $license->last_verified_at && $license->last_verified_at->gt(now()->subMinutes(30))) {
            return [
                'ok'      => $license->status === 'active',
                'status'  => $license->status,
                'message' => $license->message ?? '',
            ];
        }

        $baseUrl = config('license_client.api_base_url');
        $domain  = request()->getHost();

        try {
            $response = Http::timeout(10)
                ->acceptJson()
                ->post($baseUrl . '/license/verify', [
                    'license_key' => $license->license_key,
                    'domain'      => $domain,
                ]);

            $data = $response->json();

            if (! $response->successful() || ($data['status'] ?? null) !== 'success') {
                $license->update([
                    'status'          => $data['code'] ?? 'invalid',
                    'message'         => $data['message'] ?? 'Lisensi tidak valid.',
                    'last_verified_at'=> now(),
                ]);

                return [
                    'ok'      => false,
                    'status'  => $license->status,
                    'message' => $license->message,
                ];
            }

            $license->update([
                'status'          => $data['data']['status'] ?? 'active',
                'domain'          => $data['data']['domain'] ?? $domain,
                'message'         => 'Lisensi valid.',
                'last_verified_at'=> now(),
            ]);

            return [
                'ok'      => true,
                'status'  => $license->status,
                'message' => $license->message,
            ];
        } catch (\Throwable $e) {
            Log::error('License verify error', ['error' => $e->getMessage()]);

            return [
                'ok'      => $license->status === 'active',
                'status'  => $license->status,
                'message' => $license->message ?? 'Gagal menghubungi server lisensi.',
            ];
        }
    }
}
